<?php
session_start();

// Check if user is already logged in
if (isset($_SESSION['user_id'])) {
    header('Location: chat.php');
    exit();
}

// Check for errors
$error = '';
if (isset($_GET['error'])) {
    $error = match($_GET['error']) {
        'invalid_credentials' => 'Invalid phone number or password',
        'missing_fields' => 'Please fill in all required fields',
        'password_mismatch' => 'Passwords do not match',
        default => 'An error occurred. Please try again.'
    };
}

// Check if registration was successful
$registrationSuccess = isset($_GET['registered']) && $_GET['registered'] === '1';
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="utf-8" />
    <meta name="viewport" content="width=device-width, initial-scale=1.0, maximum-scale=1.0, user-scalable=no, viewport-fit=cover" />
    <title>Zeba Voice Project — Login</title>
    <style>
        :root {
            --bg: #000000;
            --surface: #121212;
            --muted: #2a2a2a;
            --accent-orange: #FFC929;
            --accent-yellow: #ffd33d;
            --accent-blue: #1ea7ff;
            --user-bg: #1e1e1e;
            --approved: #27ae60;
            --rejected: #e74c3c;
            --glass: rgba(255, 255, 255, 0.05);
            --text: #ffffff;
            --text-muted: #a0a0a0;
            --border: rgba(255, 201, 41, 0.15);
            --border-radius: 12px;
            --transition: all 0.3s ease;
        }

        html,body{
            height:100%;
            margin:0;
            background:var(--bg);
            color:#fff;
            font-family:Inter, "Segoe UI", Roboto, system-ui, -apple-system, "Helvetica Neue", Arial;
            -webkit-font-smoothing:antialiased;
            -moz-osx-font-smoothing:grayscale;
        }

        /* Modern Reset */
        *, *::before, *::after { box-sizing: border-box; }
        body { margin: 0; padding: 0; }
        
        /* Base Styles */
        body {
            min-height: 100vh;
            background: var(--bg);
            color: var(--text);
            font-family: 'Inter', -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, Oxygen, Ubuntu, Cantarell, sans-serif;
            line-height: 1.5;
            -webkit-font-smoothing: antialiased;
            -moz-osx-font-smoothing: grayscale;
            padding: 20px;
            display: flex;
            flex-direction: column;
        }
        
        /* Header */
        header.appbar {
            padding: 12px 20px;
            background: rgba(0, 0, 0, 0.8);
            backdrop-filter: blur(10px);
            border-bottom: 1px solid var(--border);
            display: flex;
            justify-content: space-between;
            align-items: center;
            position: fixed;
            top: 0;
            left: 0;
            right: 0;
            z-index: 1000;
        }
        
        .app-title {
            font-size: 1.1rem;
            font-weight: 700;
            color: #fff;
            display: flex;
            align-items: center;
            gap: 10px;
            padding: 8px 0;
        }
        
        .app-title .logo {
            background: linear-gradient(135deg, var(--accent-orange) 0%, var(--accent-yellow) 100%);
            color: #000;
            width: 36px;
            height: 36px;
            border-radius: 8px;
            display: flex;
            align-items: center;
            justify-content: center;
            font-size: 14px;
            font-weight: 800;
            flex-shrink: 0;
        }

        /* Main container */
        .container{
            display:flex;
            flex-direction:column;
            justify-content:center;
            align-items:center;
            padding:16px;
            height: calc(100% - 80px);
            overflow:auto;
        }

        .auth-box {
            background: var(--surface);
            border-radius: var(--border-radius);
            padding: 24px;
            width: 100%;
            max-width: 400px;
            box-shadow: 0 8px 32px rgba(0, 0, 0, 0.2);
            border: 1px solid var(--border);
            box-sizing: border-box;
            margin: 0 auto;
        }
        
        .auth-box:hover {
            transform: translateY(-2px);
            box-shadow: 0 12px 48px rgba(0, 0, 0, 0.4);
        }

        .tabs{
            display:flex;
            justify-content:space-around;
            margin-bottom:12px;
        }

        .tab{
            flex:1;
            text-align:center;
            padding:10px;
            cursor:pointer;
            font-weight:600;
            border-bottom:2px solid transparent;
            transition:border-color 0.2s;
        }

        .tab.active{
            border-color:var(--accent-yellow);
            color:var(--accent-yellow);
        }

        form{
            display:flex;
            flex-direction:column;
            gap:12px;
        }

        input, select{
            width:100%;
            padding:10px 12px;
            border-radius:8px;
            border:1px solid rgba(255,255,255,0.1);
            background:rgba(255,255,255,0.03);
            color:#fff;
            font-size:14px;
            outline:none;
            box-sizing: border-box;
        }

        input::placeholder{
            color: rgba(255,255,255,0.5);
        }

        button{
            padding:12px;
            border:none;
            border-radius:10px;
            background:var(--accent-yellow);
            font-weight:700;
            cursor:pointer;
            color:#000;
            transition:background 0.2s;
        }

        button:hover{
            background:#ffd33d;
        }

        .meta{
            font-size:13px;
            color:var(--text-muted);
            margin:8px 0 0;
            text-align:center;
            line-height:1.6;
            padding:0 8px;
        }

        .error-message {
            background: rgba(231, 76, 60, 0.1);
            color: #ff6b6b;
            padding: 14px 16px;
            border-radius: 10px;
            margin: 0 0 20px;
            border: 1px solid rgba(231, 76, 60, 0.15);
            font-size: 14px;
            line-height: 1.5;
            position: relative;
            padding-left: 44px;
        }
        
        .error-message::before {
            content: '⚠️';
            position: absolute;
            left: 16px;
            top: 50%;
            transform: translateY(-50%);
            font-size: 18px;
        }
        
        .success-message {
            background: rgba(39, 174, 96, 0.1);
            color: #6bff9a;
            padding: 14px 16px;
            border-radius: 10px;
            margin: 0 0 20px;
            border: 1px solid rgba(39, 174, 96, 0.15);
            font-size: 14px;
            line-height: 1.5;
            position: relative;
            padding-left: 44px;
        }
        
        .success-message::before {
            content: '✅';
            position: absolute;
            left: 16px;
            top: 50%;
            transform: translateY(-50%);
            font-size: 18px;
        }
        
        /* Animations */
        @keyframes fadeIn {
            from { opacity: 0; transform: translateY(10px); }
            to { opacity: 1; transform: translateY(0); }
        }
        
        .auth-box {
            animation: fadeIn 0.4s ease-out forwards;
        }

        .container {
            max-width: 100%;
            width: 100%;
            margin: 0 auto;
            padding: 20px;
            display: flex;
            flex-direction: column;
            align-items: center;
            min-height: 100vh;
            box-sizing: border-box;
            justify-content: center;
        }
        
        @media (min-width: 480px) {
            .container {
                max-width: 400px;
                margin: 0 auto;
                padding: 20px;
                min-height: 100vh;
                justify-content: center;
            }
        }

        @media(max-width:480px){
            .auth-box{ padding:16px; }
            input,select{ font-size:15px; }
            button{ font-size:16px; }
        }

        select {
            color: #fff;
            background-color: rgba(255,255,255,0.03);
        }
        
        select option {
            background-color: #121214;
            color: #fff;
        }
    </style>
</head>
<body>
<header class="appbar">
    <div class="app-title">
            <div class="logo" style="display: flex; align-items: center; justify-content: center; width: 40px; height: 40px; background: linear-gradient(135deg, var(--accent-orange), var(--accent-yellow)); border-radius: 10px; margin-right: 12px;">
                <svg width="24" height="24" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">
                    <path d="M12 2a3 3 0 0 0-3 3v6a3 3 0 0 0 6 0V5a3 3 0 0 0-3-3z" fill="#000"/>
                    <path d="M19 10v1a7 7 0 0 1-14 0v-1M12 19v3m-4 0h8" stroke="#000" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"/>
                    <path d="M12 15a2 2 0 0 1 2 2v1a2 2 0 0 1-4 0v-1a2 2 0 0 1 2-2z" fill="#4CAF50"/>
                    <path d="M12 15v-1" stroke="#4CAF50" stroke-width="2" stroke-linecap="round"/>
                </svg>
            </div>
        <span>Zeba Voice Project</span>
    </div>
</header>

<div class="container">
    <div class="auth-box" style="margin-bottom: 32px;">
        <div style="text-align: center; margin: 24px 0 32px 0;">
            <h2 style="margin:0;color:#fff;font-size:24px;font-weight:700;letter-spacing:-0.5px;margin-bottom:16px;">Zeba Voice Project</h2>
        </div>
        <?php if ($error): ?>
            <div class="error-message"><?php echo htmlspecialchars($error); ?></div>
        <?php endif; ?>
        
        <?php if ($registrationSuccess): ?>
            <div class="success-message">Registration successful! You can now log in.</div>
        <?php endif; ?>
        
        <div class="tabs">
            <div class="tab active" id="loginTab">Login</div>
            <div class="tab" id="signupTab">Sign Up</div>
        </div>

        <!-- Login Form -->
        <form id="loginForm" action="process_login.php" method="POST">
            <input type="tel" name="phone" placeholder="Phone Number" required pattern="[0-9]+" />
            <input type="password" name="password" placeholder="Password" required minlength="6" />
            <button type="submit">Login</button>
            <div class="meta" style="margin-top: 16px; text-align: center;">
                <a href="mailto:devteam@zebavoice.com" style="color: var(--accent-yellow); text-decoration: none; font-weight: 500; font-size: 14px;">
                    devteam@zebavoice.com
                </a>
            </div>
        </form>

        <!-- Sign Up Form -->
        <form id="signupForm" action="process_register.php" method="POST" style="display:none;">
            <input type="text" name="full_name" placeholder="Full Name" required />
            <input type="tel" name="phone" placeholder="Phone Number" required pattern="[0-9]+" />
            <input type="email" name="email" placeholder="Email Address" required />
            <input type="password" name="password" placeholder="Password" required minlength="6" />
            <input type="password" name="confirm_password" placeholder="Confirm Password" required minlength="6" />
            
            <select name="age_range" required>
                <option value="">Age Range</option>
                <option value="18-24">18-24</option>
                <option value="25-34">25-34</option>
                <option value="35-44">35-44</option>
                <option value="45+">45+</option>
            </select>
            
            <select name="gender" required>
                <option value="">Gender</option>
                <option value="male">Male</option>
                <option value="female">Female</option>
                <option value="other">Other</option>
                <option value="prefer_not_to_say">Prefer not to say</option>
            </select>
            
            <input type="text" name="ethnicity" placeholder="Ethnicity (optional)" />
            
            <button type="submit">Sign Up</button>
            <div class="meta">
                By signing up, you agree to our Terms of Service and Privacy Policy.
            </div>
        </form>
    </div>
    
    <div class="welcome-section" style="text-align:center;width:100%;max-width:600px;margin: 0 auto;">
        <div style="background: rgba(255, 255, 255, 0.1); padding: 20px; border-radius: 12px; text-align: left; border: 1px solid rgba(255, 201, 41, 0.2);">
            <p style="color:#fff;font-size:15px;margin:0 0 16px 0;line-height:1.7;font-weight:400;">
                Zeba Voice is an AI-driven agricultural assistant that empowers local farmers across Africa through voice technology, IoT sensors, and real-time data. By combining local languages, weather insights, and smart farming tools, Zeba helps farmers make informed decisions for better yields and sustainability.
            </p>
            <p style="color:#fff;font-size:15px;margin:0;line-height:1.7;font-weight:400;">
                Thank you for contributing your voice - every recording helps us build a smarter, more inclusive system for African agriculture. <span style="font-size: 1.2em;">🌍🇬🇭</span>
            </p>
        </div>
    </div>
</div>

<script>
const loginTab = document.getElementById('loginTab');
const signupTab = document.getElementById('signupTab');
const loginForm = document.getElementById('loginForm');
const signupForm = document.getElementById('signupForm');
const welcomeSection = document.querySelector('.welcome-section');
const urlParams = new URLSearchParams(window.location.search);
const register = urlParams.get('register');

// Show signup form if register param is set
if (register === '1') {
    signupTab.click();
}

// Tab switching
loginTab.addEventListener('click', (e) => {
    e.preventDefault();
    loginTab.classList.add('active');
    signupTab.classList.remove('active');
    loginForm.style.display = 'flex';
    signupForm.style.display = 'none';
    welcomeSection.style.display = 'block';
    // Update URL
    const url = new URL(window.location);
    url.searchParams.delete('register');
    window.history.pushState({}, '', url);
});

signupTab.addEventListener('click', (e) => {
    e.preventDefault();
    signupTab.classList.add('active');
    loginTab.classList.remove('active');
    signupForm.style.display = 'flex';
    loginForm.style.display = 'none';
    welcomeSection.style.display = 'none';
    // Update URL
    const url = new URL(window.location);
    url.searchParams.set('register', '1');
    window.history.pushState({}, '', url);
});

// Form validation
const signupFormEl = document.getElementById('signupForm');
signupFormEl.addEventListener('submit', (e) => {
    const password = signupFormEl.querySelector('input[name="password"]').value;
    const confirmPassword = signupFormEl.querySelector('input[name="confirm_password"]').value;
    
    if (password !== confirmPassword) {
        e.preventDefault();
        alert('Passwords do not match');
        return false;
    }
    
    if (password.length < 6) {
        e.preventDefault();
        alert('Password must be at least 6 characters long');
        return false;
    }
    
    return true;
});
</script>
</body>
</html>
