<?php
/**
 * Upload Configuration Checker
 * This will help diagnose upload issues
 */

error_reporting(E_ALL);
ini_set('display_errors', 1);

session_start();
?>
<!DOCTYPE html>
<html>
<head>
    <title>Upload Configuration Check</title>
    <style>
        body { font-family: system-ui; padding: 20px; background: #f5f5f5; }
        .container { max-width: 900px; margin: 0 auto; background: white; padding: 30px; border-radius: 8px; }
        h1 { color: #2d3748; }
        h2 { color: #4a5568; margin-top: 30px; border-bottom: 2px solid #e2e8f0; padding-bottom: 10px; }
        .info-table { width: 100%; border-collapse: collapse; margin: 20px 0; }
        .info-table th, .info-table td { padding: 12px; text-align: left; border-bottom: 1px solid #e2e8f0; }
        .info-table th { background: #f7fafc; font-weight: 600; width: 300px; }
        .success { color: #10b981; font-weight: bold; }
        .warning { color: #f59e0b; font-weight: bold; }
        .error { color: #ef4444; font-weight: bold; }
        .code { background: #f1f5f9; padding: 2px 6px; border-radius: 4px; font-family: monospace; }
        .section { background: #f8fafc; padding: 20px; border-radius: 6px; margin: 20px 0; }
    </style>
</head>
<body>
    <div class="container">
        <h1>🔍 Upload Configuration Check</h1>
        
        <h2>1. PHP Upload Settings</h2>
        <table class="info-table">
            <tr>
                <th>Setting</th>
                <th>Value</th>
                <th>Status</th>
            </tr>
            <tr>
                <td><strong>file_uploads</strong></td>
                <td class="code"><?php echo ini_get('file_uploads') ? 'On' : 'Off'; ?></td>
                <td><?php echo ini_get('file_uploads') ? '<span class="success">✓ OK</span>' : '<span class="error">✗ Disabled</span>'; ?></td>
            </tr>
            <tr>
                <td><strong>upload_max_filesize</strong></td>
                <td class="code"><?php echo ini_get('upload_max_filesize'); ?></td>
                <td><?php 
                    $maxSize = ini_get('upload_max_filesize');
                    $sizeInMB = (int)$maxSize;
                    echo $sizeInMB >= 10 ? '<span class="success">✓ OK</span>' : '<span class="warning">⚠ Small</span>'; 
                ?></td>
            </tr>
            <tr>
                <td><strong>post_max_size</strong></td>
                <td class="code"><?php echo ini_get('post_max_size'); ?></td>
                <td><?php 
                    $postSize = ini_get('post_max_size');
                    $sizeInMB = (int)$postSize;
                    echo $sizeInMB >= 10 ? '<span class="success">✓ OK</span>' : '<span class="warning">⚠ Small</span>'; 
                ?></td>
            </tr>
            <tr>
                <td><strong>max_execution_time</strong></td>
                <td class="code"><?php echo ini_get('max_execution_time'); ?> seconds</td>
                <td><span class="success">✓ OK</span></td>
            </tr>
            <tr>
                <td><strong>upload_tmp_dir</strong></td>
                <td class="code"><?php echo ini_get('upload_tmp_dir') ?: '/tmp'; ?></td>
                <td><?php 
                    $tmpDir = ini_get('upload_tmp_dir') ?: '/tmp';
                    echo is_writable($tmpDir) ? '<span class="success">✓ Writable</span>' : '<span class="error">✗ Not writable</span>'; 
                ?></td>
            </tr>
        </table>
        
        <h2>2. Directory Permissions</h2>
        <?php
        $uploadDir = __DIR__ . '/../uploads/';
        $exists = file_exists($uploadDir);
        $writable = $exists && is_writable($uploadDir);
        $permissions = $exists ? substr(sprintf('%o', fileperms($uploadDir)), -4) : 'N/A';
        ?>
        <table class="info-table">
            <tr>
                <th>Check</th>
                <th>Result</th>
            </tr>
            <tr>
                <td><strong>Upload Directory Path</strong></td>
                <td class="code"><?php echo htmlspecialchars($uploadDir); ?></td>
            </tr>
            <tr>
                <td><strong>Directory Exists</strong></td>
                <td><?php echo $exists ? '<span class="success">✓ Yes</span>' : '<span class="error">✗ No</span>'; ?></td>
            </tr>
            <tr>
                <td><strong>Directory Writable</strong></td>
                <td><?php echo $writable ? '<span class="success">✓ Yes</span>' : '<span class="error">✗ No</span>'; ?></td>
            </tr>
            <tr>
                <td><strong>Permissions</strong></td>
                <td class="code"><?php echo $permissions; ?></td>
            </tr>
        </table>
        
        <?php if (!$exists): ?>
        <div class="section" style="background: #fef2f2; border-left: 4px solid #ef4444;">
            <strong style="color: #ef4444;">⚠ Upload directory doesn't exist!</strong>
            <p>Run this command to create it:</p>
            <code style="display: block; background: #f1f5f9; padding: 10px; margin: 10px 0; border-radius: 4px;">
                mkdir -p <?php echo htmlspecialchars($uploadDir); ?> && chmod 755 <?php echo htmlspecialchars($uploadDir); ?>
            </code>
        </div>
        <?php elseif (!$writable): ?>
        <div class="section" style="background: #fef2f2; border-left: 4px solid #ef4444;">
            <strong style="color: #ef4444;">⚠ Upload directory is not writable!</strong>
            <p>Run this command to fix permissions:</p>
            <code style="display: block; background: #f1f5f9; padding: 10px; margin: 10px 0; border-radius: 4px;">
                chmod 755 <?php echo htmlspecialchars($uploadDir); ?>
            </code>
        </div>
        <?php endif; ?>
        
        <h2>3. Error Logs</h2>
        <div class="section">
            <p><strong>PHP Error Log:</strong> <span class="code"><?php echo ini_get('error_log') ?: 'Default system log'; ?></span></p>
            <p><strong>Custom Log File:</strong> <span class="code"><?php echo __DIR__ . '/php_errors.log'; ?></span></p>
            
            <?php
            $logFile = __DIR__ . '/../php_errors.log';
            if (file_exists($logFile)) {
                $logContent = file_get_contents($logFile);
                $lines = explode("\n", $logContent);
                $recentLines = array_slice(array_filter($lines), -20);
                
                if (!empty($recentLines)) {
                    echo '<p><strong>Recent errors (last 20 lines):</strong></p>';
                    echo '<pre style="background: #1e293b; color: #e2e8f0; padding: 15px; border-radius: 6px; overflow-x: auto; font-size: 12px;">';
                    echo htmlspecialchars(implode("\n", $recentLines));
                    echo '</pre>';
                }
            } else {
                echo '<p class="warning">⚠ No error log file found yet</p>';
            }
            ?>
        </div>
        
        <h2>4. Test Upload Form</h2>
        <div class="section">
            <p>Use this form to test file uploads:</p>
            <form method="post" enctype="multipart/form-data" style="margin-top: 20px;">
                <input type="file" name="test_file" accept="audio/*" style="padding: 10px; border: 1px solid #cbd5e0; border-radius: 4px;">
                <button type="submit" name="test_upload" style="padding: 10px 20px; background: #4a90e2; color: white; border: none; border-radius: 4px; cursor: pointer; margin-left: 10px;">Test Upload</button>
            </form>
            
            <?php
            if (isset($_POST['test_upload']) && isset($_FILES['test_file'])) {
                echo '<div style="margin-top: 20px; padding: 15px; background: #f0f9ff; border-left: 4px solid #3b82f6; border-radius: 4px;">';
                echo '<strong>Upload Test Results:</strong><br><br>';
                echo '<strong>File Name:</strong> ' . htmlspecialchars($_FILES['test_file']['name']) . '<br>';
                echo '<strong>File Size:</strong> ' . number_format($_FILES['test_file']['size']) . ' bytes<br>';
                echo '<strong>File Type:</strong> ' . htmlspecialchars($_FILES['test_file']['type']) . '<br>';
                echo '<strong>Error Code:</strong> ' . $_FILES['test_file']['error'];
                
                if ($_FILES['test_file']['error'] === UPLOAD_ERR_OK) {
                    echo '<br><span class="success">✓ File uploaded successfully to temp location!</span>';
                    
                    // Try to move the file
                    if (!$exists) {
                        mkdir($uploadDir, 0755, true);
                    }
                    $testFile = $uploadDir . 'test_' . time() . '_' . basename($_FILES['test_file']['name']);
                    if (move_uploaded_file($_FILES['test_file']['tmp_name'], $testFile)) {
                        echo '<br><span class="success">✓ File moved successfully to: ' . htmlspecialchars($testFile) . '</span>';
                        chmod($testFile, 0644);
                    } else {
                        echo '<br><span class="error">✗ Failed to move file to destination</span>';
                    }
                } else {
                    $errors = [
                        UPLOAD_ERR_INI_SIZE => 'File exceeds upload_max_filesize',
                        UPLOAD_ERR_FORM_SIZE => 'File exceeds MAX_FILE_SIZE',
                        UPLOAD_ERR_PARTIAL => 'File was only partially uploaded',
                        UPLOAD_ERR_NO_FILE => 'No file was uploaded',
                        UPLOAD_ERR_NO_TMP_DIR => 'Missing temporary folder',
                        UPLOAD_ERR_CANT_WRITE => 'Failed to write file to disk',
                        UPLOAD_ERR_EXTENSION => 'Extension blocked the upload',
                    ];
                    echo '<br><span class="error">✗ Upload failed: ' . ($errors[$_FILES['test_file']['error']] ?? 'Unknown error') . '</span>';
                }
                echo '</div>';
            }
            ?>
        </div>
        
        <div style="margin-top: 30px; padding-top: 20px; border-top: 2px solid #e2e8f0;">
            <a href="mini_task_manager.php" style="color: #4a90e2; text-decoration: none; font-weight: 600;">← Back to Mini Task Manager</a>
        </div>
    </div>
</body>
</html>
